import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;


public class JLancraft implements Runnable {
	public static final int WAR3_UDP_PORT = 6112;
	public static final int WAR3_DEFAULT_VERSION = 24;
	
	// These threads must only be started/stopped while the current thread
	// holds the lock on the JLancraft object.
	private Thread jlancraftThread;
	private Thread hostCheckerThread;
	private Thread datagramSenderThread;
	private Map<Integer, Thread> proxyListeners;
	private Thread[] proxyPipes;
	
	// incomingGameinfos maps the hash of an incoming GAMEINFO packet an index
	// in the outgoingGameinfos list.  It is used to determine whether a
	// received GAMEINFO has been seen already and to remove GAMEINFOS that
	// are no longer being sent by the host. NOTE: The actual packet data
	// stored in outgoingGameinfos may not have the same hash as is used as
	// a key in incomingGameinfos due to port changes in the packet data.
	private Map<Integer, Integer> incomingGameinfos;
	
	// This list contains a set of byte arrays to send as datagrams to
	// Warcraft (that is, to localhost:6112).  This list should only be
	// accessed by the thread that holds its lock.
	private List<byte[]> outgoingGameinfos;
	
	private PrintStream out;
	private boolean error;

	private String hostname;
	private int[] checkPorts;
	private int gameVersion;
	private Map<Integer, Integer> portMappings;
	
	public static void main(String args[]) {
//		JLancraft jlc2 = new JLancraft("67.9.159.140", 24, new int[] {6112, 6111});
		JLancraft jlc2 = new JLancraft("localhost", 24, new int[] {6112, 6111});
//		jlc2.portMappings.put(6113, 14333);
		jlc2.run();
	}

	public JLancraft(String hostname) {
		this(hostname, WAR3_DEFAULT_VERSION, new int[] {WAR3_UDP_PORT}, System.out, null);
	}
	public JLancraft(String hostname, int gameVersion) {
		this(hostname, gameVersion, new int[] {WAR3_UDP_PORT}, System.out, null);
	}
	public JLancraft(String hostname, int gameVersion, int[] checkPorts) {
		this(hostname, gameVersion, checkPorts, System.out, null);
	}
	public JLancraft(String hostname, int gameVersion, int[] checkPorts,
			PrintStream out) {
		this(hostname, gameVersion, checkPorts, out, null);
	}
	
	public JLancraft(String hostname, int gameVersion, int[] checkPorts,
			PrintStream out, int[][] portMappings) {
		super();
		this.hostname = hostname;
		this.gameVersion = gameVersion;
		if (checkPorts != null)
			this.checkPorts = Arrays.copyOf(checkPorts, checkPorts.length);
		else
			this.checkPorts = new int[0];
		this.out = out;
		this.portMappings = new HashMap<Integer, Integer>();
		
		if (portMappings != null) {
			for (int[] mapping : portMappings) {
				this.portMappings.put(mapping[0], mapping[1]);
			}
		}
		
		this.incomingGameinfos = new HashMap<Integer, Integer>();
		this.outgoingGameinfos = new LinkedList<byte[]>();
		this.proxyListeners = new HashMap<Integer, Thread>();
		this.proxyPipes = new Thread[2];
	}

	public void run() {
		println("Running...");
		
		if (jlancraftThread != null) {
			println("Error: Cannot run multiple threads using the same JLancraft object");
			println("Exiting with error");
			return;
		}
		
		jlancraftThread = Thread.currentThread();
		
		InetAddress hostAddr;
		try {
			hostAddr = InetAddress.getByName(hostname);
		} catch (UnknownHostException e) {
			println("Error: Unknown host: " + hostname);
			return;
		}
		
		
		while (!Thread.currentThread().isInterrupted() && !error) {
			out.println();
			println("Starting host checker and datagram sender...");
			
			stopAllThreads();
			
			synchronized (outgoingGameinfos) {
				incomingGameinfos.clear();
				outgoingGameinfos.clear();
			}
			
			startHostChecker(hostAddr);
			startDatagramSender();

			println("Waiting for threads to exit...");
			try {
				hostCheckerThread.join();
				
				if (datagramSenderThread != null)
					datagramSenderThread.join();
				
				if (proxyPipes != null) {
					if (proxyPipes[0] != null)
						proxyPipes[0].join();
					if (proxyPipes[1] != null)
						proxyPipes[1].join();
				}
			} catch (InterruptedException e) {
				println("Interrupted");
				break;
			}

			println("All threads have exited");
			println("Restarting...");
		}
		
		if (error) {
			println("Exiting with error");
		} else {
			println("Exiting normally");
		}
		
		stopAllThreads();
	}
	
	private synchronized void startHostChecker(InetAddress addr) {
		if (Thread.currentThread().isInterrupted())
			return;
		
		if (hostCheckerThread != null && hostCheckerThread.isAlive())
			stopHostChecker();
		
		Runnable r = new HostChecker(addr);
		hostCheckerThread = new Thread(r, "HostChecker");
		hostCheckerThread.start();
	}
	
	private synchronized void startDatagramSender() {
		if (Thread.currentThread().isInterrupted())
			return;
		
		if (datagramSenderThread != null && datagramSenderThread.isAlive())
			stopDatagramSender();
		
		Runnable r = new DatagramSender();
		datagramSenderThread = new Thread(r, "DatagramSender");
		datagramSenderThread.start();
	}
	
	private synchronized void startProxyListener(ServerSocket socket, InetSocketAddress dest) {
		if (Thread.currentThread().isInterrupted())
			return;
		
		int port = socket.getLocalPort();
		
		if (proxyListeners.containsKey(port)) {
			Thread t = proxyListeners.get(port);
			if (t != null && t.isAlive())
				stopProxyListener(port);
		}
		
		Runnable r = new ProxyListener(socket, dest);
		Thread t = new Thread(r, "ProxyListener-" + port + "-" + dest.getPort());
		proxyListeners.put(port, t);
		t.start();
	}
	
	private synchronized void startProxyPipes(InputStream in1, OutputStream out1, InputStream in2, OutputStream out2) {
		if (Thread.currentThread().isInterrupted())
			return;
		
		stopProxyPipes();
		
		Runnable a = new ProxyPipe(in1, out2);
		Runnable b = new ProxyPipe(in2, out1);
		
		proxyPipes[0] = new Thread(a, "ProxyPipeA");
		proxyPipes[1] = new Thread(b, "ProxyPipeB");
		
		proxyPipes[0].start();
		proxyPipes[1].start();
		
	}
	
	private synchronized void stopHostChecker() {
		if (hostCheckerThread != null)
			hostCheckerThread.interrupt();
	}
	
	private synchronized void stopDatagramSender() {
		if (datagramSenderThread != null)
			datagramSenderThread.interrupt();
	}
	
	private synchronized void stopProxyListener(int port) {
		Thread t = proxyListeners.get(port);
		if (t != null) {
			t.interrupt();
			proxyListeners.remove(port);
		}
	}

	private synchronized void stopAllProxyListeners() {
		for (Thread t : proxyListeners.values()) {
			t.interrupt();
		}

		proxyListeners.clear();
	}
	
	private synchronized void stopProxyPipes() {
		if (proxyPipes[0] != null)
			proxyPipes[0].interrupt();
		if (proxyPipes[1] != null)
			proxyPipes[1].interrupt();
	}
	
	private synchronized void stopAllThreads() {
		stopHostChecker();
		stopDatagramSender();
		stopAllProxyListeners();
		stopProxyPipes();
	}
	
	private synchronized boolean setupProxy(Socket a, Socket b) {
		InputStream inA, inB;
		OutputStream outA, outB;
		try {
			inA = a.getInputStream();
			outA = a.getOutputStream();
			
			inB = b.getInputStream();
			outB = b.getOutputStream();
		} catch (IOException e) {
			println("Error: Received IOException while setting up proxy");
			println(e.toString());
			return false;
		}
		
		startProxyPipes(inA, outA, inB, outB);
		
		stopHostChecker();
		stopDatagramSender();
		stopAllProxyListeners();
		
		return true;
	}
	
	private void println(String s) {
		out.println("[" + Thread.currentThread().getName() + "] " + s);
	}
	
	private void printf(String fmt, Object... params) {
		out.print("[" + Thread.currentThread().getName() + "] ");
		out.printf(fmt, params);
	}
	
	private void abortWithError() {
		println("Exiting with error");
		error = true;
		jlancraftThread.interrupt();
	}
	
	private int getGamePort(byte[] data) {
		int len = data.length;
		int ret = ((data[len - 1] & 0xFF) << 8) | (data[len - 2] & 0xFF);
		return ret;
	}
	
	private void setGamePort(byte[] data, int port) {
		int len = data.length;
		data[len-1] = (byte)((port >> 8) & 0xFF);
		data[len-2] = (byte)(port & 0xFF);
	}
	
	private byte[] getSearchgameData() {

		byte[] ret = {
			(byte)247,					// W3GS_HEADER_CONSTANT
			47,							// W3GS_SEARCHGAME
			16, 0,						// total packet length (little-endian)
			80, 88, 51, 87,				// product ID ("W3XP", reversed);
			(byte)gameVersion, 0, 0, 0,	// warcraft minor version (little-endian)
			0, 0, 0, 0,					// unknown
		};
		
		return ret;
	}
	
	private class HostChecker implements Runnable {
		private InetAddress hostAddr;
		public static final int MAX_SEND_FAILURES = 2;
		public static final int MAX_RECV_FAILURES = 2;
		public static final int CHECK_INTERVAL = 5000;

		public HostChecker(InetAddress hostAddr) {
			super();
			this.hostAddr = hostAddr;
		}
		
		public void run() {
			println("Running...");
			
			DatagramSocket sock;
			try {
				sock = new DatagramSocket();
			} catch (IOException e) {
				println("Error: Caught exception when creating DatagramSocket");
				println(e.toString());
				abortWithError();
				return;
			}
			
			byte[] data = getSearchgameData();
			DatagramPacket pkt = new DatagramPacket(data, data.length);
			pkt.setAddress(hostAddr);
			
			byte[] buf = new byte[2048];
			DatagramPacket pktIn = new DatagramPacket(buf, buf.length);
			
			int sendFailCount = 0;
			int recvFailCount = 0;
			int remainingTime;
			long startTime, endTime;
			
			println("Checking for games every " + CHECK_INTERVAL + " ms...");
			
			while (!Thread.currentThread().isInterrupted()) {
				if (!sendSearchgames(sock, pkt)) {
					sendFailCount++;
					if (sendFailCount > MAX_SEND_FAILURES) {
						println("Error: Failed to send any SEARCHGAMEs in the past " + sendFailCount + " attempts");
						sock.close();
						abortWithError();
						return;
					}
				} else {
					sendFailCount = 0;
				}
				
				remainingTime = CHECK_INTERVAL;
				
				
				while (remainingTime > 0) {
					try {
						sock.setSoTimeout(remainingTime);
						
						startTime = System.currentTimeMillis();
						sock.receive(pktIn);
						handleReply(pktIn);
						endTime = System.currentTimeMillis();
						
						remainingTime -= (endTime - startTime);
						recvFailCount = 0;
					} catch (SocketTimeoutException e) {
						remainingTime = 0;
						recvFailCount = 0;
					} catch (InterruptedIOException e) {
						remainingTime = 0;
						Thread.currentThread().interrupt();
						break;
					} catch (IOException e) {
						println("Warning: Exception occurred while waiting for GAMEINFO replies");
						println(e.toString());
						recvFailCount++;
						
						if (recvFailCount > MAX_RECV_FAILURES) {
							println("Error: Failed to listen for packets for the past " + recvFailCount + " attempts");
							sock.close();
							abortWithError();
							return;
						}
					}
				}
				
				if (remainingTime > 0) {
					try {
						Thread.sleep(remainingTime);
					} catch (InterruptedException e) {
						break;
					}
				}
			}

			sock.close();
			println("Exiting normally");
		}
		
		private boolean sendSearchgames(DatagramSocket sock, DatagramPacket pkt) {
			boolean success = false;
			for (int i=0;i<checkPorts.length;i++) {
				pkt.setPort(checkPorts[i]);
				try {
					sock.send(pkt);
					success = true;
				} catch (IOException e) {
					println("Warning: Failed to send SEARCHGAME to port " + checkPorts[i]);
				}
			}
			return success;
		}
		
		private void handleReply(DatagramPacket pkt) {
			if (!pkt.getAddress().equals(hostAddr)) {
				println("Warning: Ignoring incoming packet from host " + pkt.getAddress() + " (expected " + hostAddr + ")");
				return;
			}
			
			if (pkt.getLength() < 6) {
				println("Warning: Packet from " + pkt.getAddress() + " is too short");
				return;
			}
			
			// Get the packet data
			byte[] data = new byte[pkt.getLength()];
			System.arraycopy(pkt.getData(), pkt.getOffset(), data, 0, pkt.getLength());
			
			// Check if this GAMEINFO has been seen before
			// Zero the uptime field because it changes from one GAMEINFO to 
			// the next.
			
			byte[] uptime = new byte[4];
			System.arraycopy(data, data.length-6, uptime, 0, 4);
			Arrays.fill(data, data.length-6, data.length-2, (byte)0);
			
			int hash = Arrays.hashCode(data);
			
			System.arraycopy(uptime, 0, data, data.length-6, 4);
			
			
			// Add this packet's info to incomingGameinfos/outgoingGameinfos
			synchronized (outgoingGameinfos) {
			
				if (incomingGameinfos.containsKey(hash))
					return;
	
				String hashStr = Integer.toHexString(hash);
				println("Found new game with GAMEINFO hash 0x" + Integer.toHexString(hash));
				
				// Get the game port
				int gamePort = getGamePort(data);
				int listenPort = gamePort;
				
				println("Game 0x" + hashStr + ": GAMEINFO specifies gameport " + gamePort);
				
				if (portMappings.containsKey(gamePort) && portMappings.get(gamePort) != null) {
					println("Gameport " + gamePort + " has been redirected to port " + portMappings.get(gamePort));
					gamePort = portMappings.get(gamePort);
				}
				
				// Start a ServerSocket for a ProxyListener to listen on
				ServerSocket server = null;
				
				try {
					server = new ServerSocket(0, 0, InetAddress.getLocalHost());
				} catch (IOException e) {
					println("Error: Unable to bind to any TCP port");
					abortWithError();
					return;
				}
				listenPort = server.getLocalPort();
				println("Game 0x" + hashStr + ": Listening on TCP port " + listenPort);
				
				setGamePort(data, listenPort);
				
				
				println("Game 0x" + hashStr + ": Starting proxy listener for TCP port " + listenPort);
				println("Game 0x" + hashStr + ": Connections will be forwarded to remote port " + gamePort);
				
				// Start the ProxyListener for this game
				startProxyListener(server, new InetSocketAddress(hostAddr, gamePort));
			
				int idx = outgoingGameinfos.size();
				outgoingGameinfos.add(data);
				incomingGameinfos.put(hash, idx);
			}
		}
	}

	private class DatagramSender implements Runnable {
		public static final int MAX_SEND_FAILURES = 2;
		public static final int DATAGRAM_INTERVAL = 1000;
		
		public void run() {
			println("Running...");
			
			DatagramSocket sock;
			
			try {
				sock = new DatagramSocket();
			} catch (IOException e) {
				println("Error: Caught exception while creating DatagramSocket");
				println(e.toString());
				abortWithError();
				return;
			}
			
			println("Running on UDP port " + sock.getLocalPort());
			
			byte[] buf = new byte[0];
			DatagramPacket pkt = new DatagramPacket(buf, 0);
			try {
				pkt.setAddress(InetAddress.getLocalHost());
			} catch (UnknownHostException e) {
				println("Error: Unable to set datagram destination to local host");
				println(e.toString());
				sock.close();
				abortWithError();
				return;
			}
			pkt.setPort(WAR3_UDP_PORT);
			
			boolean success;
			int failCount = 0;
			
			while (!Thread.currentThread().isInterrupted()) {
				success = false;
				synchronized (outgoingGameinfos) {
					if (outgoingGameinfos.size() == 0)
						success = true;
					for (byte[] data : outgoingGameinfos) {
						pkt.setData(data);
						try {
							sock.send(pkt);
							success = true;
						} catch (IOException e) {
							println("Warning: Failed to send datagram");
							println(e.toString());
						}
					}
				}
				
				if (!success) {
					failCount++;
					if (failCount > MAX_SEND_FAILURES) {
						println("Error: Unable to send any datagrams in the past " + failCount + " attempts");
						abortWithError();
						sock.close();
						return;
					}
				} else {
					failCount = 0;
				}
				
				try {
					Thread.sleep(DATAGRAM_INTERVAL);
				} catch (InterruptedException e) {
					break;
				}
			}

			sock.close();
			println("Exiting normally");
		}
	}
	
	private class ProxyListener implements Runnable {
		public static final int ACCEPT_TIMEOUT = 5000;
		
		ServerSocket server;
		InetSocketAddress dest;

		public ProxyListener(ServerSocket server, InetSocketAddress dest) {
			super();
			this.server = server;
			this.dest = dest;
		}
		
		public void run() {
			println("Running...");
			
			try {
				server.setSoTimeout(ACCEPT_TIMEOUT);
			} catch (IOException e) {
				println("Caught exception when setting accept timeout");
				println(e.toString());
				printErrorAndRespawn();
			}
			
			Socket incoming = null;
			while (!Thread.currentThread().isInterrupted() && incoming == null) {
				try {
					incoming = server.accept();
				} catch (SocketTimeoutException e) {
					// try again
				} catch (IOException e) {
					println("Error: Failed to accept connection");
					println(e.toString());
					printErrorAndRespawn();
					return;
				}
			}
			
			if (Thread.currentThread().isInterrupted()) {
				println("Exiting normally");
				return;
			}
			
			println("Received incoming connection from " + incoming.getRemoteSocketAddress());
			
			Socket outgoing;
			try {
				outgoing = new Socket(dest.getAddress(), dest.getPort());
			} catch (IOException e) {
				println("Error: Failed to connect to destination: " + dest);
				println(e.toString());
				println("Closing incoming connection...");
				try {
					incoming.close();
				} catch (IOException e2) {
					println("Caught exception while closing incoming connection");
					println(e2.toString());
				}
				printErrorAndRespawn();
				return;
			}
			
			println("Created outgoing connection to " + outgoing.getRemoteSocketAddress());
			
			if (setupProxy(incoming, outgoing))
				println("Exiting normally");
			else
				printErrorAndRespawn();
			
		}
		
		private void printErrorAndRespawn() {
			println("Exiting with error");
			println("Respawning...");
			startProxyListener(server, dest);
		}
	}
	
	private class ProxyPipe implements Runnable {
		private InputStream inStream;
		private OutputStream outStream;
		
		public ProxyPipe(InputStream inStream, OutputStream outStream) {
			super();
			this.inStream = inStream;
			this.outStream = outStream;
		}
		
		public void run() {
			println("Running...");
			
			byte[] buf = new byte[4096];
			int len;
			boolean error = false;
			
			while (!Thread.currentThread().isInterrupted()) {
				try {
					len = inStream.read(buf);
					if (len == -1)
						break;
					outStream.write(buf, 0, len);
				} catch (InterruptedIOException e) {
					break;
				} catch (IOException e) {
					println("Warning: Caught exception while transferring data");
					println(e.toString());
					error = true;
					break;
				}
			}
			
			try {
				outStream.close();
			} catch (IOException e) {
			}
			
			try {
				inStream.close();
			} catch (IOException e) {
			}
			
			if (error)
				println("Exiting due to exception");
			else
				println("Exiting normally");
			
		}
	}
}
