import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.prefs.Preferences;

import javax.swing.GroupLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.GroupLayout.Alignment;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;


public class JLancraftGUI extends JFrame implements ActionListener {
	
	private static final String VERSION = "0.3.0";

        private static final int MAX_HISTORY = 10;

	private static final Preferences PREFS = Preferences.userRoot().node("JLancraftGUI");
	
	private JDialog advancedSettingsDialog;
	
	private JComboBox hostnameCombo;
	private JTextArea outputArea;
	
	private JScrollPane outputScroller;
	
	private JButton startButton;
	private JButton stopButton;
	private JButton helpButton;
	private JButton advancedSettingsButton;
	

	private JTextField udpPortsField;
	private JTextField gameVersionField;
	private JTable portRedirectTable;

	private JScrollPane portRedirectScroller;
	
	private JButton addRedirectButton;
	private JButton removeRedirectButton;
	private JButton okButton;
	private JButton cancelButton;
	
	private PortRedirectTableModel portRedirectTableModel;
	
	private PrintStream out;
	private Thread pipeReaderThread;
	
	private Thread jlancraftThread;
	
	
	private String hostname;
	private int gameVersion = 24;
	private int[] udpPorts = { 6112 };
	private int[][] portRedirects = new int[0][];
	
	public static void main(String[] args) {
		Runnable r = new Runnable() {
			public void run() {
				JFrame f = new JLancraftGUI();
				f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
				f.setLocationByPlatform(true);
				f.setVisible(true);
			}
		};
		SwingUtilities.invokeLater(r);
	}
	
	public JLancraftGUI() {
		super("JLancraft " + VERSION);
		buildMainGUI();
		buildSettingsGUI();
		startPipeThread();
		loadSettings();
	}
	
	private void startPipeThread() {
		PipedInputStream pipeIn;
		PipedOutputStream pipeOut;
		try {
			pipeIn = new PipedInputStream();
			pipeOut = new PipedOutputStream(pipeIn);
		} catch (IOException e) {
			appendOutput("[" + Thread.currentThread().getName() + "] Error: Unable to create pipes for output\n");
			appendOutput("[" + Thread.currentThread().getName() + "] " + e + "\n");
			return;
		}
		
		Runnable r = new PipeReader(pipeIn);
		pipeReaderThread = new Thread(r, "OutputPrinter");
		pipeReaderThread.start();
		
		out = new PrintStream(pipeOut);
	}
	
	private void buildMainGUI() {
		JLabel hostnameLabel = new JLabel("Hostname:");

		hostnameCombo = new JComboBox();
		
		startButton = new JButton("Start");
		stopButton = new JButton("Stop");
		helpButton = new JButton("Help");
		advancedSettingsButton = new JButton("Advanced Settings...");

		outputArea = new JTextArea(15, 40);
		outputScroller = new JScrollPane(outputArea, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

		startButton.addActionListener(this);
		stopButton.addActionListener(this);
		helpButton.addActionListener(this);
		advancedSettingsButton.addActionListener(this);
		advancedSettingsButton.setActionCommand("Settings");
		
                hostnameCombo.setEditable(true);
		stopButton.setEnabled(false);
		
		GroupLayout layout = new GroupLayout(getContentPane());
		setLayout(layout);
		
		layout.setHorizontalGroup(
		layout.createParallelGroup()
			.addGroup(
			layout.createSequentialGroup()
				.addGap(5)
				.addComponent(hostnameLabel)
				.addGap(5)
				.addComponent(hostnameCombo)
				.addGap(5)
				.addComponent(startButton)
				.addGap(5)
				.addComponent(stopButton)
				.addGap(5)
			)
			.addGroup(
			layout.createSequentialGroup()
				.addGap(0, 5, Short.MAX_VALUE)
				.addComponent(advancedSettingsButton)
				.addGap(5)
				.addComponent(helpButton)
				.addGap(0, 5, Short.MAX_VALUE)
			)
			.addComponent(outputScroller)
		);
		
		layout.setVerticalGroup(
		layout.createSequentialGroup()
			.addGap(5)
			.addGroup(
			layout.createParallelGroup(GroupLayout.Alignment.BASELINE)
				.addComponent(hostnameLabel)
				.addComponent(hostnameCombo)
				.addComponent(startButton)
				.addComponent(stopButton)
			)
			.addGap(5)
			.addGroup(
			layout.createParallelGroup(GroupLayout.Alignment.BASELINE)
				.addComponent(advancedSettingsButton)
				.addComponent(helpButton)
			)
			.addGap(5)
			.addComponent(outputScroller)
		);
					
		
		pack();
	}
	
	private void buildSettingsGUI() {
		advancedSettingsDialog = new JDialog(this, "Advanced Settings", true);
		
		JLabel udpPortsLabel = new JLabel("UDP Port(s):");
		JLabel gameVersionLabel = new JLabel("Game Minor Version:");
		JLabel portRedirectLabel = new JLabel("Game Port Redirections:");

		udpPortsField = new JTextField("6112", 10);
		gameVersionField = new JTextField("24", 2);
		addRedirectButton = new JButton("Add");
		removeRedirectButton = new JButton("Remove");
		okButton = new JButton("OK");
		cancelButton = new JButton("Cancel");

		createPortRedirectTable();
		portRedirectScroller = new JScrollPane(portRedirectTable);
		portRedirectScroller.setPreferredSize(new Dimension(300,200));

		addRedirectButton.addActionListener(this);
		removeRedirectButton.addActionListener(this);
		okButton.addActionListener(this);
		cancelButton.addActionListener(this);
		
		GroupLayout layout = new GroupLayout(advancedSettingsDialog.getContentPane());
		advancedSettingsDialog.getContentPane().setLayout(layout);
		
		layout.setHorizontalGroup(
		layout.createParallelGroup()
			.addGroup(
			layout.createSequentialGroup()
				.addGap(5)
				.addGroup(
				layout.createParallelGroup()
					.addComponent(udpPortsLabel)
					.addComponent(gameVersionLabel)
				)
				.addGap(5)
				.addGroup(
				layout.createParallelGroup()
					.addComponent(udpPortsField)
					.addComponent(gameVersionField)
				)
				.addGap(5)
			)
			.addGroup(
			layout.createSequentialGroup()
				.addGap(5)
				.addGroup(
				layout.createParallelGroup()
					.addComponent(portRedirectLabel)
					.addGroup(
					layout.createSequentialGroup()
						.addComponent(portRedirectScroller)
						.addGap(5)
						.addGroup(
						layout.createParallelGroup()
							.addComponent(addRedirectButton)
							.addComponent(removeRedirectButton)
						)
					)
				)
				.addGap(5)
			)
			.addGroup(
			layout.createSequentialGroup()
				.addGap(5, 5, Short.MAX_VALUE)
				.addComponent(okButton)
				.addGap(5)
				.addComponent(cancelButton)
				.addGap(5)
			)
		);
		
		layout.setVerticalGroup(
		layout.createSequentialGroup()
			.addGap(5)
			.addGroup(
			layout.createParallelGroup(Alignment.BASELINE)
				.addComponent(udpPortsLabel)
				.addComponent(udpPortsField)
			)
			.addGap(5)
			.addGroup(
			layout.createParallelGroup(Alignment.BASELINE)
				.addComponent(gameVersionLabel)
				.addComponent(gameVersionField)
			)
			.addGap(15)
			.addComponent(portRedirectLabel)
			.addGap(5)
			.addGroup(
			layout.createParallelGroup()
				.addComponent(portRedirectScroller)
				.addGroup(
				layout.createSequentialGroup()
					.addComponent(addRedirectButton)
					.addGap(5)
					.addComponent(removeRedirectButton)
				)
			)
			.addGap(10)
			.addGroup(
			layout.createParallelGroup()
				.addComponent(okButton)
				.addComponent(cancelButton)
			)
			.addGap(5)
		);
		
		advancedSettingsDialog.pack();
			
	}
	
	private void createPortRedirectTable() {
		portRedirectTableModel = new PortRedirectTableModel();
		portRedirectTable = new JTable(portRedirectTableModel);
	}
	
	private class PortRedirectTableModel extends AbstractTableModel {
		private static final long serialVersionUID = -9125638713537846044L;
		
		private List<int[]> redirs = new ArrayList<int[]>();
		
		public String getColumnName(int columnIndex) {
			if (columnIndex == 0)
				return "Game Port";
			if (columnIndex == 1)
				return "Redirected Port";
			return "";
		}
		
		public Class<?> getColumnClass(int columnIndex) {
			if (columnIndex == 0 || columnIndex == 1)
				return Integer.class;
			return null;
		}
		
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return rowIndex >= 0 && rowIndex < redirs.size()
					&& columnIndex >= 0 && columnIndex < 2;
		}
		
		public void setValueAt(Object value, int rowIndex, int columnIndex) {
			if (!isCellEditable(rowIndex, columnIndex))
				return;
			if (!(value instanceof Integer))
				return;
			redirs.get(rowIndex)[columnIndex] = (Integer)value;
			fireTableCellUpdated(rowIndex, columnIndex);
		}
		
		public int getRowCount() {
			return redirs.size();
		}
		
		public int getColumnCount() {
			return 2;
		}
		
		public Object getValueAt(int rowIndex, int columnIndex) {
			return redirs.get(rowIndex)[columnIndex];
		}
		
		public int[] getMapping(int index) {
			return Arrays.copyOf(redirs.get(index), 2);
		}
		
		public void addMapping(int from, int to) {
			redirs.add(new int[] { from, to });
			fireTableRowsInserted(redirs.size()-1, redirs.size()-1);
		}
		
		public void removeMapping(int rowIdx) {
			redirs.remove(rowIdx);
			fireTableRowsDeleted(rowIdx, rowIdx);
		}
		
		public void removeMappings(int[] rowIdxs) {
			for (int i = rowIdxs.length - 1; i >= 0; i--) {
				removeMapping(rowIdxs[i]);
			}
		}
		
		public void clearMappings() {
			int size = redirs.size();
			redirs.clear();
			if (size > 0)
				fireTableRowsDeleted(0, size - 1);
		}
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();
		
		if (cmd.equals("Start")) {
			startJLancraft();
		} else if (cmd.equals("Stop")) {
			stopJLancraft();
		} else if (cmd.equals("Settings")) {
			displayAdvancedSettingsDialog();
		} else if (cmd.equals("Help")) {
			displayHelp();
		} else if (cmd.equals("Add")) {
			addPortRedirect();
		} else if (cmd.equals("Remove")) {
			removePortRedirect();
		} else if (cmd.equals("OK")) {
			closeAdvancedSettingsDialog(true);
		} else if (cmd.equals("Cancel")) {
			closeAdvancedSettingsDialog(false);
		}
	}
	
	private synchronized void startJLancraft() {
		if (jlancraftThread != null) {
			out.println("[" + Thread.currentThread().getName() + "] Error: A JLancraft thread is already running\n");
			return;
		}
		
		hostname = (String)hostnameCombo.getSelectedItem();
		saveSettings();
		
		Runnable r = new JLancraft(hostname, gameVersion, Arrays.copyOf(udpPorts, udpPorts.length), out, portRedirects);
		jlancraftThread = new Thread(r, "JLancraft");

		out.println("[" + Thread.currentThread().getName() + "] Starting JLancraft thread...\n");
		jlancraftThread.start();
		
		Runnable r2 = new Runnable() {
			public void run() {
				try {
					jlancraftThread.join();
				} catch (InterruptedException e) {}
				stopJLancraft();
			}
		};
		new Thread(r2).start();

		startButton.setEnabled(false);
		stopButton.setEnabled(true);
	}
	
	private synchronized void stopJLancraft() {
		if (jlancraftThread == null) {
			return;
		}
		
		out.println("[" + Thread.currentThread().getName() + "] Stopping JLancraft thread...\n");
		jlancraftThread.interrupt();
		jlancraftThread = null;
		
		startButton.setEnabled(true);
		stopButton.setEnabled(false);
	}
	
	private void displayAdvancedSettingsDialog() {
		StringBuffer sb = new StringBuffer();
		for (int port : udpPorts) {
			sb.append(port);
			sb.append(", ");
		}
		sb.delete(sb.length() - 2, sb.length());
		
		udpPortsField.setText(sb.toString());
		
		gameVersionField.setText("" + gameVersion);
		
		portRedirectTableModel.clearMappings();
		for (int[] redir : portRedirects) {
			portRedirectTableModel.addMapping(redir[0], redir[1]);
		}
		
		
		Rectangle mainWinBounds = this.getBounds();
		int centerX = mainWinBounds.x + mainWinBounds.width / 2;
		int centerY = mainWinBounds.y + mainWinBounds.height / 2;
		Rectangle advSetWinBounds = advancedSettingsDialog.getBounds();
		advSetWinBounds.x = centerX - advSetWinBounds.width / 2;
		advSetWinBounds.y = centerY - advSetWinBounds.height / 2;
		advancedSettingsDialog.setBounds(advSetWinBounds);
		
		
		advancedSettingsDialog.setVisible(true);
	}
	
	private void displayHelp() {
		clearOutput();
		appendOutput("No help available in this version");
	}
	
	private void addPortRedirect() {
		portRedirectTableModel.addMapping(JLancraft.WAR3_UDP_PORT, JLancraft.WAR3_UDP_PORT);
		int rowIdx = portRedirectTableModel.getRowCount() - 1;
		portRedirectTable.editCellAt(rowIdx, 0);
	}
	
	private void removePortRedirect() {
		portRedirectTableModel.removeMappings(portRedirectTable.getSelectedRows());
	}
	
	private void closeAdvancedSettingsDialog(boolean save) {
		if (save) {
			if (!validateAdvancedSettings())
				return;
			copyAdvancedSettings();
			saveAdvancedSettings();
		}
		advancedSettingsDialog.setVisible(false);
	}
	
	private boolean validateAdvancedSettings() {
		String portStr = udpPortsField.getText();
		String[] portArr = portStr.split(",");
		int port;
		for (String str : portArr) {
			str = str.trim();
			try {
				port = Integer.parseInt(str);
			} catch (NumberFormatException e) {
				JOptionPane.showMessageDialog(advancedSettingsDialog, "Invalid UDP port: \"" + str + "\" is not a number", "Error", JOptionPane.ERROR_MESSAGE);
				udpPortsField.requestFocus();
				return false;
			}
			
			if (port <= 0 || port >= 65536) {
				JOptionPane.showMessageDialog(advancedSettingsDialog, "Invalid UDP port: " + port + " is not within the range 1-65535", "Error", JOptionPane.ERROR_MESSAGE);
				udpPortsField.requestFocus();
				return false;
			}
		}
		
		int version;
		String versionStr = gameVersionField.getText();
		try {
			version = Integer.parseInt(versionStr);
		} catch (NumberFormatException e) {
			JOptionPane.showMessageDialog(advancedSettingsDialog, "Invalid game minor version: \"" + versionStr + "\" is not a number", "Error", JOptionPane.ERROR_MESSAGE);
			gameVersionField.requestFocus();
			return false;
		}
		
		return true;
	}
	
	private void copyAdvancedSettings() {
		String[] portStrArray = udpPortsField.getText().split(",");
		udpPorts = new int[portStrArray.length];
		for (int i=0;i<portStrArray.length;i++) {
			udpPorts[i] = Integer.parseInt(portStrArray[i].trim());
		}
		
		gameVersion = Integer.parseInt(gameVersionField.getText());
		
		int numRedirs = portRedirectTableModel.getRowCount();
		portRedirects = new int[numRedirs][2];
		for (int i=0;i<numRedirs;i++) {
			portRedirects[i] = Arrays.copyOf(portRedirectTableModel.getMapping(i), 2);
		}
	}
	
	private void saveSettings() {
                PREFS.put("lastHostname", hostname);

                if (hostnameCombo.getSelectedIndex() != -1)
                    hostnameCombo.removeItemAt(hostnameCombo.getSelectedIndex());

                hostnameCombo.insertItemAt(hostname, 0);
                hostnameCombo.setSelectedIndex(0);

                int hostnameCount = Math.min(hostnameCombo.getItemCount(), MAX_HISTORY);
                PREFS.putInt("hostnameCount", hostnameCount);
                for (int i = 0; i < hostnameCount; i++) {
                    PREFS.put("hostname" + i, (String)hostnameCombo.getItemAt(i));
                }

		saveAdvancedSettings();
	}
	
	private void saveAdvancedSettings() {
		StringBuffer sb = new StringBuffer();
		for (int port : udpPorts) {
			sb.append(port);
			sb.append(',');
		}
		PREFS.put("udpPorts", sb.toString());

		PREFS.putInt("gameVersion", gameVersion);
		
		sb = new StringBuffer();
		for (int[] redir : portRedirects) {
			sb.append(redir[0]);
			sb.append(':');
			sb.append(redir[1]);
			sb.append(',');
		}
		PREFS.put("portRedirects", sb.toString());
	}
	
	private void loadSettings() {
		hostname = PREFS.get("lastHostname", "localhost");
                int hostnameCount = PREFS.getInt("hostnameCount", 0);

                hostnameCombo.removeAllItems();
                for (int i=0;i<hostnameCount;i++) {
                    String currentHost = PREFS.get("hostname" + i, "");
                    if (currentHost.length() > 0)
                        hostnameCombo.addItem(currentHost);
                }

                hostnameCombo.setSelectedItem(hostname);

		// remaining fields will be set in displayAdvancedSettingsDialog()
		
		String[] udpPortStrArray = PREFS.get("udpPorts", "" + JLancraft.WAR3_UDP_PORT).split(",");
		int[] tempUDPPorts = new int[udpPortStrArray.length];
		try {
			for (int i=0;i<tempUDPPorts.length;i++) {
				tempUDPPorts[i] = Integer.parseInt(udpPortStrArray[i].trim());
			}
			udpPorts = tempUDPPorts;
		} catch (NumberFormatException e) {
			// use default for udpPorts
			out.println("[" + Thread.currentThread().getName() + "] Warning: Unable to load UDP Ports setting");
		}
		
		gameVersion = PREFS.getInt("gameVersion", JLancraft.WAR3_DEFAULT_VERSION);
		
		String redirStr = PREFS.get("portRedirects", "");
		if (redirStr.trim().length() > 0) {
			String[] redirStrArray = redirStr.split(",");
			int[][] tempPortRedirects = new int[redirStrArray.length][2];
			try {
				String[] redirPorts;
				for (int i=0;i<tempPortRedirects.length;i++) {
					redirPorts = redirStrArray[i].split(":");
					for (int j=0;j<2;j++) {
						tempPortRedirects[i][j] = Integer.parseInt(redirPorts[j].trim());
					}
				}
				
				portRedirects = tempPortRedirects;
			} catch (NumberFormatException e) {
				// use default for portRedirects
				out.println("[" + Thread.currentThread().getName() + "] Warning: Unable to load Port Redirects setting");
			} catch (ArrayIndexOutOfBoundsException e) {
				// use default for portRedirects
				out.println("[" + Thread.currentThread().getName() + "] Warning: Unable to load Port Redirects setting");
			}
		}
	}

	private void appendOutput(final String str) {
		Runnable r = new Runnable() {
			public void run() {
				outputArea.append(str);
				outputArea.setCaretPosition(outputArea.getText().length());
			}
		};

		if (SwingUtilities.isEventDispatchThread()) {
			r.run();
		} else {
			SwingUtilities.invokeLater(r);
		}
	}
	
	private void clearOutput() {
		Runnable r = new Runnable() {
			public void run() {
				outputArea.setText("");
			}
		};

		if (SwingUtilities.isEventDispatchThread()) {
			r.run();
		} else {
			SwingUtilities.invokeLater(r);
		}
	}
	
	private class PipeReader implements Runnable {
		private InputStream in;

		public PipeReader(InputStream in) {
			this.in = in;
		}

		public void run() {
			appendOutput("[" + Thread.currentThread().getName() + "] Running...\n");
			
			int len;
			byte[] buf = new byte[4096];

			while (!Thread.currentThread().isInterrupted()) {
				try {
					len = in.read(buf);
				} catch (IOException e) {
					if (e.getMessage() != null && e.getMessage().equals("Write end dead")) {
						try {
							Thread.sleep(50);
						} catch (InterruptedException e2) {
							break;
						}
						continue;
					} else {
						appendOutput("[" + Thread.currentThread().getName() + "] Error: Received exception while handling output\n");
						appendOutput("[" + Thread.currentThread().getName() + "] " + e + "\n");
						appendOutput("[" + Thread.currentThread().getName() + "] Exiting with error\n");
						return;
					}
				}
				if (len == -1)
					break;
				appendOutput(new String(buf, 0, len));
			}
			appendOutput("[" + Thread.currentThread().getName() + "] Exiting normally\n");
		}
	}
}
